<?php

namespace Modules\User\Http\Controllers;

use Illuminate\Http\Request;
use Modules\Common\Entities\Animal;
use Modules\Common\Entities\RegisteredAnimal;
use Modules\Common\Entities\Report;
use Modules\Common\Http\Requests\ReportDeleteRequest;
use Modules\Common\Http\Requests\ReportStoreRequest;
use Modules\Common\Http\Requests\ReportUpdateRequest;
use Modules\User\Http\Controllers\Traits\ImagesUploadable;


class ReportController extends AuthenticatedController
{
    use ImagesUploadable;

    public function index()
    {
        // Get reports of the current user
        $query = $this->user->reportsWithMembers()->sortable(['expires_at' => 'desc']);

        // Check if user want to see deleted reports
        $showDeleted = request()->has('showDeleted');
        if ($showDeleted) {
            $query = $query->onlyTrashed();
        }

        // Execute query
        $reports = $query->get();


        // Return View
        return view('user::reports.index', compact('reports', 'showDeleted'));
    }

    public function show(int $reportId)
    {
        // Get report
        $report = Report::withTrashed()->findOrFail($reportId);

        // Authorize user
        $this->authorize('write', $report);

        // Return View
        return view('user::reports.show', compact('report'));
    }

    public function edit(Report $report)
    {
        // Authorize user
        $this->authorize('write', $report);

        // Get values
        $animalSpecies = Animal::getAnimalSpeciesList();
        $animalConditions = Animal::ANIMAL_CONDITIONS;
        $identificationTypes = Animal::IDENTIFICATION_TYPES;

        // Return View
        return view('user::reports.edit', compact(
            'report',
            'animalSpecies',
            'animalConditions',
            'identificationTypes',
        ));
    }

    public function update(ReportUpdateRequest $request, Report $report)
    {
        // Authorize user
        $this->authorize('write', $report);

        // Update resource
        $validated = $request->validated();
        $report->update($validated);

        // Handle notifies
        $report->handleNotifies();

        // Redirect
        return redirect()
            ->route('user.reports.show', $report)
            ->with('success', __('messages.edit_report.success'));
    }

    public function remove(Report $report)
    {
        // Authorize user
        $this->authorize('write', $report);

        // Get values
        $deletionReasons = Report::DELETION_REASONS;

        // Return View
        return view('user::reports.remove', compact(
            'report',
            'deletionReasons',
        ));
    }

    public function delete(ReportDeleteRequest $request,  Report $report)
    {
        // Authorize user
        $this->authorize('write', $report);

        // Handle deletion reason
        $validated = $request->validated();
        $validated['deletion_reason'] = $validated['deletion_reason'] ?? Report::DELETION_REASON_OTHER;
        $report->update($validated);

        // Delete resource
        $report->delete();

        // Redirect
        return redirect()
            ->route('user.reports.index', $report)
            ->with('success', __('messages.delete_report.success'));
    }

    public function create(Request $request)
    {
        // Get user
        $user = $request->user();

        // Get registered animals
        $registeredAnimals = $user->registeredAnimals()->take(50)->get();

        // Get registered animal if exists
        $registeredAnimalId = $request->get('registered_animal_id');
        $registeredAnimal = $user->registeredAnimals()->find($registeredAnimalId);
        if (!$registeredAnimal) {
            $registeredAnimal = new RegisteredAnimal();
        }

        // Get thumbnails
        $thumbnails = $registeredAnimal->thumbnails;

        // Get values
        $animalSpecies = Animal::getAnimalSpeciesList();
        $identificationTypes = Animal::IDENTIFICATION_TYPES;

        // Return View
        return view('user::reports.create', compact(
            'registeredAnimals',
            'registeredAnimal',
            'thumbnails',
            'animalSpecies',
            'identificationTypes',
        ));
    }

    public function store(ReportStoreRequest $request)
    {
        // Create resource
        $validated = $request->validated();
        $validated['type'] = Report::TYPE_LOST;
        $validated['animal_condition'] = Animal::ANIMAL_CONDITION_HEALTHY;
        $report = auth()->user()->reports()->create($validated);

        // Set expiration
        $report->setExpiration();

        // Handle images
        $this->handleImages($request, $report);

        // Refresh resource
        $report->refresh();

        // Handle notifies
        $report->handleNotifies(true);

        // Redirect
        return redirect()
            ->route('user.reports.index')
            ->with('success', __('messages.create_report.success'));
    }

    public function extendExpiration(Report $report)
    {
        // Authorize user
        $this->authorize('write', $report);

        // Update resource
        $report->extendExpiration();

        // Redirect
        return redirect()
            ->route('user.reports.index', $report)
            ->with('success', __('messages.extend_expiration.success'));
    }

    public function suspend(Report $report)
    {
        // Authorize user
        $this->authorize('write', $report);

        // Update resource
        $report->suspend();

        // Redirect
        return redirect()
            ->route('user.reports.index', $report)
            ->with('success', __('messages.suspend.success'));
    }

    public function unsuspend(Report $report)
    {
        // Authorize user
        $this->authorize('write', $report);

        // Update resource
        $report->unsuspend();

        // Redirect
        return redirect()
            ->route('user.reports.index', $report)
            ->with('success', __('messages.unsuspend.success'));
    }

    // End of class
}
