<?php

namespace Modules\User\Http\Controllers;

use Modules\Common\Entities\Animal;
use Modules\Common\Entities\RegisteredAnimal;
use Modules\Common\Http\Requests\RegisteredAnimalStoreRequest;
use Modules\Common\Http\Requests\RegisteredAnimalUpdateRequest;
use Modules\User\Http\Controllers\Traits\ImagesUploadable;

class RegisteredAnimalController extends AuthenticatedController
{

    use ImagesUploadable;

    public function index()
    {
        // Get registeredAnimals of the current user
        $registeredAnimals = $this->user->registeredAnimals()->get();

        // Return View
        return view('user::registered_animals.index', compact('registeredAnimals'));
    }

    public function show(RegisteredAnimal $registeredAnimal)
    {
        // Authorize user
        $this->authorize('write', $registeredAnimal);

        // Return View
        return view('user::registered_animals.show', compact('registeredAnimal'));
    }

    public function edit(RegisteredAnimal $registeredAnimal)
    {
        // Authorize user
        $this->authorize('write', $registeredAnimal);

        // Get values
        $animalSpecies = Animal::getAnimalSpeciesList();
        $animalConditions = Animal::ANIMAL_CONDITIONS;
        $identificationTypes = Animal::IDENTIFICATION_TYPES;

        // Get thumbnails
        $thumbnails = $registeredAnimal->thumbnails;

        // Return View
        return view('user::registered_animals.edit', compact(
            'registeredAnimal',
            'thumbnails',
            'animalSpecies',
            'identificationTypes',
        ));
    }

    public function update(RegisteredAnimalUpdateRequest $request, RegisteredAnimal $registeredAnimal)
    {
        // Authorize user
        $this->authorize('write', $registeredAnimal);

        // Update resource
        $validated = $request->validated();
        $registeredAnimal->update($validated);

        // Handle images
        $this->handleImages($request, $registeredAnimal);

        // Redirect
        return redirect()
            ->route('user.registered_animals.show', $registeredAnimal)
            ->with('success', __('messages.edit_registered_animal.success'));
    }

    public function delete(RegisteredAnimal $registeredAnimal)
    {
        // Authorize user
        $this->authorize('write', $registeredAnimal);

        // Update resource
        $registeredAnimal->delete();

        // Redirect
        return redirect()
            ->route('user.registered_animals.index', $registeredAnimal)
            ->with('success', __('messages.delete_registered_animal.success'));
    }

    public function create()
    {
        // Get values
        $animalSpecies = Animal::getAnimalSpeciesList();
        $identificationTypes = Animal::IDENTIFICATION_TYPES;

        // Return View
        return view('user::registered_animals.create', compact(
            'animalSpecies',
            'identificationTypes',
        ));
    }

    public function store(RegisteredAnimalStoreRequest $request)
    {
        // Create resource
        $validated = $request->validated();
        $registeredAnimal = auth()->user()->registeredAnimals()->create($validated);

        // Handle images
        $this->handleImages($request, $registeredAnimal);

        // Refresh resource
        $registeredAnimal->refresh();

        // Redirect
        return redirect()
            ->route('user.registered_animals.index')
            ->with('success', __('messages.create_registered_animal.success'));
    }

    // End of class
}
