<?php

namespace Modules\User\Http\Controllers;

use Modules\Common\Entities\User;
use Modules\User\Http\Requests\MemberStoreRequest;
use Modules\User\Http\Requests\MemberUpdateRequest;
use Modules\User\Notifications\MemberCredentialsNotification;

use Hash;
use Str;


class MemberController extends AuthenticatedController
{

    public function index()
    {
        // Get members of the current organization
        $members = $this->user->members()->sortable()->withCount('reports')->get();

        // Return View
        return view('user::members.index', compact('members'));
    }

    public function show(User $member)
    {
        // Authorize user
        $this->authorize('write-member', $member);

        // Return View
        return view('user::members.show', compact('member'));
    }

    public function edit(User $member)
    {
        // Authorize user
        $this->authorize('write-member', $member);

        // Return View
        return view('user::members.edit', compact('member'));
    }

    public function update(MemberUpdateRequest $request, User $member)
    {
        // Authorize user
        $this->authorize('write-member', $member);

        // Update resource
        $validated = $request->validated();
        $member->update($validated);

        // Redirect
        return redirect()
            ->route('user.members.show', $member)
            ->with('success', __('messages.edit_member.success'));
    }

    public function delete(User $member)
    {
        // Authorize user
        $this->authorize('write-member', $member);

        // Attach reports from member to organization
        $organizationId = $member->organization->id;
        $member->reports->each->update(['user_id' => $organizationId]);

        // Delete resource
        $member->forceDelete();

        // Redirect
        return redirect()
            ->route('user.members.index', $member)
            ->with('success', __('messages.delete_member.success'));
    }

    public function create()
    {
        // Return View
        return view('user::members.create');
    }

    public function store(MemberStoreRequest $request)
    {
        // Validate data
        $validated = $request->validated();

        // Generate password for member
        $password = Str::random(10);
        $validated['password'] = Hash::make($password);

        // Set account type to Access
        $validated['account_type'] = User::ACCOUNT_TYPE_ACCESS;

        // create member
        $member = User::create($validated);

        // Verify email of member
        $member->markEmailAsVerified();

        // Verify phone of member
        $member->markPhoneAsVerified(null);

        // Attach member to organization
        $member->forceFill(['extras->organization_id' => $this->user->id])->save();

        // Notify member on its credentials
        $member->notify(new MemberCredentialsNotification($member, $password, $this->user->organizationName));

        // Redirect
        return redirect()
            ->route('user.members.index')
            ->with('success', __('messages.create_member.success'));
    }

    // End of class
}
