<?php

namespace Modules\User\Http\Controllers\Auth;

use Illuminate\Http\Request;
use Illuminate\Http\Response;

use Twilio\Rest\Client as TwilioClient;

use App\Http\Controllers\Controller;
use Illuminate\Validation\ValidationException;


class VerificationPhoneController extends Controller
{
    /*
      |-------------------------------------------------------------------------
      | Phone Verification Controller
      |-------------------------------------------------------------------------
      |
      | This controller is responsible for handling phone verification for any
      | user that recently registered with the application. Phone code may also
      | be re-sent if the user didn't receive the original code.
      |
     */

    private $client;

    public function __construct()
    {
        $this->middleware('throttle:6,1')->only('verify', 'resend');

        // Init. Twilio Client
        $accountSid = getenv("TWILIO_ACCOUNT_SID");
        $authToken = getenv("TWILIO_AUTH_TOKEN");
        $this->client = new TwilioClient($accountSid, $authToken);
    }

    protected function redirectPath()
    {
        return route('user.profile.index');
    }

    public function show(Request $request)
    {
        return $request->user()->hasVerifiedPhone()
            ? redirect($this->redirectPath())
            : view('user::auth.verify_phone');
    }

    public function resend(Request $request)
    {
        // Redirect user if phone is already verified
        if ($request->user()->hasVerifiedPhone()) {
            return redirect($this->redirectPath());
        }

        // Validate data
        $request->validate([
            'phone' => ['required', 'phone:AUTO,mobile', 'max:20'],
        ]);

        // Send SMS with verification code
        $valid = $this->resendVerificationCode($request['phone']);

        // Failed
        if (!$valid) {
            abort(503);
        }

        // Success
        return $request->wantsJson()
            ? new Response('', 204)
            : redirect()->back()->withInput($request->input())->with('resent_code', true);
    }

    public function verify(Request $request)
    {
        $user = $request->user();

        // Redirect user if phone is already verified
        if ($user->hasVerifiedPhone()) {
            return redirect($this->redirectPath());
        }

        // Validate data
        $request->validate([
            'phone' => ['required', 'phone:AUTO,mobile', 'max:20'],
            'verification_code' => ['required', 'digits:6'],
        ]);

        // Get field values
        $phone = $request['phone'];
        $code = $request['verification_code'];

        // Send SMS with verification code
        $valid = $this->checkVerificationCode($phone, $code);

        // Failed
        if (!$valid) {
            throw ValidationException::withMessages([
                'verification_code' => __('validation.phone_code', [
                    'attribute' => __('validation.attributes.verification_code'),
                ]),
            ]);
        }

        // Success
        $user->markPhoneAsVerified($phone);

        auth('user')->login($user);

        return $request->wantsJson()
            ? new Response('', 204)
            : redirect($this->redirectPath())->with('verified_phone', true);
    }

    private function resendVerificationCode($phone)
    {
        try {
            $verificationSid = getenv("TWILIO_VERIFICATION_SID");
            $this->client->verify->v2->services($verificationSid)->verifications->create($phone, 'sms');
        } catch (\Throwable $e) {
            return false;
        }

        return true;
    }

    private function checkVerificationCode($phone, $code)
    {
        try {
            $verificationSid = getenv("TWILIO_VERIFICATION_SID");
            $check = $this->client->verify->v2->services($verificationSid)->verificationChecks->create("$code", ['to' => $phone]);

            if ($check->status != 'approved') {
                throw new \Exception();
            }
        } catch (\Throwable $e) {
            return false;
        }

        return true;
    }

    // End of class
}
