<?php

namespace Modules\User\Http\Controllers\Auth;

use Illuminate\Auth\Events\Registered;
use Illuminate\Http\Response;
use Illuminate\Foundation\Auth\RegistersUsers;
use Illuminate\Routing\Controller;

use Modules\Common\Entities\User;
use Modules\User\Http\Requests\RegisterRequest;

use Auth;
use Hash;

class RegisterController extends Controller
{
    /*
      |-------------------------------------------------------------------------
      | Register Controller
      |-------------------------------------------------------------------------
      |
      | This controller handles the registration of new users as well as their
      | validation and creation. By default this controller uses a trait to
      | provide this functionality without requiring any additional code.
      |
     */

    use RegistersUsers;

    public function __construct()
    {
        $this->middleware('guest:user');
    }

    protected function redirectTo()
    {
        return route('user.profile.index');
    }

    protected function guard()
    {
        return Auth::guard('user');
    }

    protected function showRegisterForm()
    {
        return view('user::auth.register');
    }

    public function register(RegisterRequest $request)
    {
        event(new Registered($user = $this->create($request->all())));

        $this->guard()->login($user);

        if ($response = $this->registered($request, $user)) {
            return $response;
        }

        return $request->wantsJson()
            ? new Response('', 201)
            : redirect($this->redirectPath());
    }

    protected function create(array $data)
    {
        // Check if it is an organization
        $isOrganization = $data['account_type'] == User::ACCOUNT_TYPE_ORGANIZATION;

        // Prepare data
        $validated = [
            'email' => $data['email'],
            'password' => Hash::make($data['password']),
            'first_name' => $data['first_name'],
            'last_name' => $data['last_name'],
            'gender' => $data['gender'],
            'date_of_birth' => $data['date_of_birth'],
            'address' => $data['address'],
            'location_latitude' => $data['location_latitude'],
            'location_longitude' => $data['location_longitude'],
            'location_zipcode' => $data['location_zipcode'],
            'location_country' => $data['location_country'],
            'phone' => $data['phone'],
            'account_type' => $data['account_type'],
            'is_verified' => !$isOrganization,
        ];

        // Add extras data for organization account
        if ($isOrganization) {
            $validated['extras->organization_function'] = $data['organization_function'];
            $validated['extras->organization_name'] = $data['organization_name'];
            $validated['extras->organization_number'] = $data['organization_number'];
        }

        // Create user
        return User::create($validated);
    }

    // End of class
}
