<?php

namespace Modules\Reports\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Routing\Controller;

use Modules\Common\Entities\Report;

class ReportsController extends Controller
{

    /***************************************************************************
     * Methods
     **************************************************************************/
    public function index(Request $request)
    {
        // Get reports
        $query = Report::available()->latest('expires_at');

        // Apply Location Filter
        $query = $this->applyLocationFilter($request->all(), $query);

        // Apply Type Filter
        $query = $this->applyTypeFilter($request->all(), $query);

        // Apply animal condition filter
        $query = $this->applyAnimalConditionFilter($request->all(), $query);

        // Apply AnimalSpecies Filter
        $query = $this->applyAnimalSpeciesFilter($request->all(), $query);

        // Get reports
        $reports = $query->paginate(12);

        // Keep get parameters
        $reports->appends($request->all());

        // Return view
        return view('reports::index', compact('reports'));
    }

    public function show($id)
    {
        // Get report
        $report = Report::find($id);

        // Get latest reports
        $latestReports = Report::available()->take(4)->latest('expires_at')->get();

        // Redirect to custom 404 if not exists
        if (!$report) {
            return view('reports::404', compact('latestReports'));
        }

        // Get author
        $author = $report->user->organization;

        // Return view
        return view('reports::show', compact('report', 'author', 'latestReports'));
    }

    /***************************************************************************
     * Utils
     **************************************************************************/
    private function applyLocationFilter(array $attributes, $query)
    {
        // Get location
        $address = $attributes['address'] ?? null;
        $latitude = $attributes['latitude'] ?? null;
        $longitude = $attributes['longitude'] ?? null;
        $radius = $attributes['radius'] ?? null;

        // If exists then apply filter
        if ($address && $latitude && $longitude && $radius) {
            $query = $query->inRadius([
                'latitude' => $latitude,
                'longitude' => $longitude,
            ], $radius);
        }

        // Return query
        return $query;
    }

    private function applyTypeFilter(array $attributes, $query)
    {
        // Get type
        $type = $attributes['type'] ?? null;

        // If exists then apply filter
        if ($type) {
            $query = $query->where('reports.type', '=', $type);
        }

        // Return query
        return $query;
    }

    private function applyAnimalConditionFilter(array $attributes, $query)
    {
        // Get animal condition
        $animalCondition = $attributes['animal_condition'] ?? null;

        // If exists then apply filter
        if ($animalCondition) {
            $query = $query->where('reports.animal_condition', '=', $animalCondition);
        }

        // Return query
        return $query;
    }

    private function applyAnimalSpeciesFilter(array $attributes, $query)
    {
        // Get species
        $species = $attributes['animal_species'] ?? null;

        // If exists then apply filter
        if ($species) {
            $query = $query->where('reports.animal_species', 'like', "$species%");
        }

        // Return query
        return $query;
    }

    // End of class
}
