<?php

namespace Modules\Common\Entities;

use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;

use Laravel\Passport\HasApiTokens;
use Backpack\CRUD\app\Models\Traits\CrudTrait;
use Illuminate\Database\Eloquent\SoftDeletes;
use Kyslik\ColumnSortable\Sortable;
use Modules\Common\Entities\Contracts\MustVerifyPhone;
use Modules\Common\Entities\Presenters\UserPresenter;
use Modules\Common\Entities\Relationships\UserRelationships;
use Modules\Common\Entities\Scopes\UserScopes;
use Modules\Common\Entities\Traits\MustVerifyPhone as TraitsMustVerifyPhone;
use Modules\User\Notifications\CustomPasswordResetNotification;
use Modules\User\Notifications\DeleteAccountConfirmation;

use Carbon\Carbon;


class User extends Authenticatable implements MustVerifyEmail, MustVerifyPhone
{

    /***************************************************************************
     * Traits
     **************************************************************************/
    use SoftDeletes,
        HasApiTokens,
        TraitsMustVerifyPhone,
        Notifiable,
        Sortable,
        CrudTrait,
        UserRelationships,
        UserScopes,
        UserPresenter;

    /***************************************************************************
     * Attributes
     **************************************************************************/
    protected $guard = 'user';

    protected $fakeColumns = [
        'extras',
    ];

    protected $fillable = [
        'email', 'password', 'first_name', 'last_name', 'gender',
        'date_of_birth', 'phone', 'phone_is_private', 'address',
        'location_latitude', 'location_longitude', 'location_zipcode',
        'location_country', 'account_type', 'is_verified',
        'email_verified_at', 'phone_verified_at', 'extras',
        'extras->organization_id', 'extras->organization_function',
        'extras->organization_name', 'extras->organization_number',
        'deleted_at', 'force_premium', 'is_blocked',
    ];

    protected $hidden = [
        'password', 'remember_token', 'email_verified_at', 'phone_verified_at',
    ];

    protected $casts = [
        'email_verified_at' => 'datetime',
        'phone_verified_at' => 'datetime',
        'phone_is_private' => 'int',
        'is_organization' => 'int',
        'is_verified' => 'int',
        'is_blocked' => 'int',
        'is_premium' => 'int',
        'extras' => 'array',
    ];

    protected $sortable = [
        'email', 'first_name', 'last_name', 'created_at',
    ];

    protected $sortableAs = [
        'reports_count'
    ];

    /***************************************************************************
     * FCM Notification
     **************************************************************************/
    public function routeNotificationForFcm()
    {
        return $this->notificationTokens()->pluck('token')->toArray();
    }

    /***************************************************************************
     * Phone verification
     **************************************************************************/
    public function sendPhoneVerificationCode()
    {
    }

    /***************************************************************************
     * Forgot password
     **************************************************************************/
    public function sendPasswordResetNotification($token)
    {
        $this->notify(new CustomPasswordResetNotification($token));
    }

    /***************************************************************************
     * Accessors
     **************************************************************************/
    public function getCreatedAtAttribute($value)
    {
        return Carbon::parse($value)->timezone('Europe/Brussels');
    }

    /***************************************************************************
     * Additional attributes
     **************************************************************************/
    public function getIsDeletedAttribute()
    {
        return $this->deleted_at != null;
    }

    public function getHasVerifiedEmailAttribute()
    {
        return $this->hasVerifiedEmail();
    }

    public function getHasVerifiedPhoneAttribute()
    {
        return $this->hasVerifiedPhone();
    }

    public function getIsIndividualAttribute()
    {
        return $this->account_type == self::ACCOUNT_TYPE_INDIVIDUAL;
    }

    public function getIsOrganizationAttribute()
    {
        return $this->account_type == self::ACCOUNT_TYPE_ORGANIZATION;
    }

    public function getIsOrganizationAccessAttribute()
    {
        return $this->account_type == self::ACCOUNT_TYPE_ACCESS;
    }

    public function getOrganizationIdAttribute()
    {
        try {
            return $this->extras['organization_id'];
        } catch (\Throwable $e) {
            return null;
        }
    }

    public function getOrganizationNameAttribute()
    {
        try {
            return $this->organization->extras['organization_name'];
        } catch (\Throwable $e) {
            return null;
        }
    }

    public function getOrganizationNumberAttribute()
    {
        try {
            return $this->organization->extras['organization_number'];
        } catch (\Throwable $e) {
            return null;
        }
    }

    public function getOrganizationFunctionAttribute()
    {
        try {
            return $this->organization->extras['organization_function'];
        } catch (\Throwable $e) {
            return null;
        }
    }

    public function getIsPremiumAttribute()
    {
        if ($this->force_premium) {
            return 1;
        }
        return PremiumCity::inZipcodes($this->location_zipcode)
            ->whereCountry($this->location_country)
            ->exists() ? 1 : 0;
    }

    /***************************************************************************
     * Methods
     **************************************************************************/
    public function generateAccessToken()
    {
        $tokenResult = $this->createToken('');
        $tokenResult->token->save();

        return $tokenResult;
    }

    public function sendDeleteAccountConfirmationNotification()
    {
        $this->notify(new DeleteAccountConfirmation());
    }

    public function hasEmail()
    {
        return !empty($this->email);
    }

    public function hasPhone()
    {
        return $this->phone_is_private == 0 && !empty($this->phone);
    }

    public function hasAddress()
    {
        return !empty($this->address) && !empty($this->location_zipcode) && !empty($this->location_country);
    }

    public function forceDeleteWithMembers()
    {
        $this->members()->forceDelete();
        $this->forceDelete();
    }

    /***************************************************************************
     * Account Type Values
     **************************************************************************/
    const ACCOUNT_TYPE_INDIVIDUAL = 'individual';
    const ACCOUNT_TYPE_ORGANIZATION = 'organization';
    const ACCOUNT_TYPE_ACCESS = 'access';

    const TYPES = [
        self::ACCOUNT_TYPE_INDIVIDUAL,
        self::ACCOUNT_TYPE_ORGANIZATION,
        self::ACCOUNT_TYPE_ACCESS,
    ];

    /***************************************************************************
     * Gender Values
     **************************************************************************/
    const GENDER_MALE = 'm';
    const GENDER_FEMALE = 'f';
    const GENDER_UNSPECIFIED = 'u';

    const GENDERS = [
        self::GENDER_MALE,
        self::GENDER_FEMALE,
        self::GENDER_UNSPECIFIED,
    ];


    /***************************************************************************
     * Default Values
     **************************************************************************/
    const DEFAULT_VALUES = [
        'first_name' => 'John',
        'last_name' => 'Doe',
    ];

    // End of class
}
