<?php

namespace Modules\Common\Entities;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

use Backpack\CRUD\app\Models\Traits\CrudTrait;
use Spatie\MediaLibrary\Models\Media;
use Spatie\MediaLibrary\HasMedia\HasMedia;
use Spatie\MediaLibrary\HasMedia\HasMediaTrait;
use Kyslik\ColumnSortable\Sortable;

use Modules\Common\Entities\Presenters\AnimalPresenter;
use Modules\Common\Entities\Presenters\ReportPresenter;
use Modules\Common\Entities\Relationships\ReportRelationships;
use Modules\Common\Entities\Scopes\ReportScopes;
use Modules\Common\Entities\Traits\Localizable;

use Carbon\Carbon;
use Carbon\CarbonInterval;
use Illuminate\Support\Facades\Notification;
use Modules\Common\Notifications\NotifyUserIfLostAnimalExists;
use Modules\Common\Notifications\NotifyUserIfLostAnimalWasFound;
use Modules\Common\Notifications\NotifyUserIfReportMatchesAlert;

class Report extends Model implements HasMedia
{

    /***************************************************************************
     * Traits
     **************************************************************************/
    use SoftDeletes,
        ReportRelationships,
        ReportScopes,
        AnimalPresenter,
        ReportPresenter,
        HasMediaTrait,
        CrudTrait,
        Sortable,
        Localizable;

    /***************************************************************************
     * Attributes
     **************************************************************************/
    protected $fillable = [
        'user_id', 'location_address', 'location_latitude',
        'location_longitude', 'location_zipcode', 'location_country', 'type',
        'animal_species', 'animal_condition', 'identification_type',
        'identification_number', 'final_location', 'comment', 'state',
        'deleted_at', 'deletion_reason', 'deletion_reason_comment',
    ];

    protected $hidden = [
        'user_id',
    ];

    protected $dates = [
        'expires_at',
    ];

    public $sortable = [
        'user_id', 'type', 'animal_species', 'created_at', 'expires_at',
    ];

    /***************************************************************************
     * Setters
     **************************************************************************/
    public function setIdentificationNumberAttribute($identificationNumber)
    {
        $this->attributes['identification_number'] = strtoupper(preg_replace("/[^a-zA-Z0-9]+/", "", $identificationNumber));
    }

    /***************************************************************************
     * Getters
     **************************************************************************/
    public function getCreatedAtAttribute($value)
    {
        return Carbon::parse($value)->timezone('Europe/Brussels');
    }

    public function getLinkAttribute()
    {
        return route('reports.show', [$this]);
    }

    public function getHasExpiredAttribute()
    {
        return $this->hasExpired();
    }

    /***************************************************************************
     * Methods
     **************************************************************************/
    public function setExpiration()
    {
        $interval = $this->getExpirationInterval();
        $this->expires_at = $this->created_at->copy()->add($interval);
        $this->save();
    }

    public function extendExpiration()
    {
        $interval = $this->getExpirationInterval();
        $this->expires_at = Carbon::now()->add($interval);
        $this->save();
    }

    public function suspend()
    {
        $this->state = self::STATE_SUSPENDED;
        $this->save();
    }

    public function unsuspend()
    {
        $this->state = self::STATE_WAITING;
        $this->save();
    }

    public function isAvailable(): bool
    {
        return !$this->hasWaitingAbuses() && $this->isWaiting();
    }

    public function hasIncompleteAddress(): bool
    {
        return !$this->location_address || !$this->location_zipcode || !$this->location_country;
    }

    public function hasWaitingAbuses(): bool
    {
        return $this->abuses()->waiting()->exists();
    }

    public function isDone(): bool
    {
        return $this->state == self::STATE_DONE;
    }

    public function isWaiting(): bool
    {
        return $this->state == self::STATE_WAITING;
    }

    public function isSuspended(): bool
    {
        return $this->state == self::STATE_SUSPENDED;
    }

    public function isFound(): bool
    {
        return $this->type == self::TYPE_FOUND;
    }

    public function isLost()
    {
        return $this->type == self::TYPE_LOST;
    }

    public function isHealthy(): bool
    {
        return $this->animal_condition == Animal::ANIMAL_CONDITION_HEALTHY;
    }

    public function isNotHealthy(): bool
    {
        return !$this->isHealthy();
    }

    public function hasExpired(): bool
    {
        return $this->expires_at < Carbon::now();
    }

    public function handleNotifies($force = false)
    {
        // Notifiy alerts
        if ($force || $this->wasChanged('animal_species')) {
            $this->notifyAlerts();
        }

        // Notify matching id
        if ($force || $this->wasChanged('identification_number')) {
            $this->notifyIdentificationNumberMatching();
        }
    }

    public function notifyIdentificationNumberMatching()
    {
        // Init. vars
        $matchingReport = null;

        // Handle only if field is not null
        if ($this->identification_number) {
            if ($this->isFound()) {
                $matchingReport = Report::lost()->waiting()->latest()->where('identification_number', $this->identification_number)->first();
            } else {
                $matchingReport = Report::found()->waiting()->latest()->where('identification_number', $this->identification_number)->first();
            }
        }

        // Notify user if matching has been found
        if ($matchingReport) {
            if ($this->isFound()) {
                $matchingReport->user->notify(new NotifyUserIfLostAnimalWasFound($this));
                $this->user->notify(new NotifyUserIfLostAnimalExists($matchingReport));
            } else {
                $this->user->notify(new NotifyUserIfLostAnimalWasFound($matchingReport));
                $matchingReport->user->notify(new NotifyUserIfLostAnimalExists($this));
            }
        }
    }

    public function notifyAlerts()
    {
        // Notify only animals found
        if ($this->isLost()) return;

        // Retrieve all users setted alerts that matches the report
        $users = User::whereHas('alerts', function ($query) {
            $query->matchesReport($this);
        })->get();

        // Notify users
        Notification::send($users, new NotifyUserIfReportMatchesAlert($this));
    }

    /***************************************************************************
     * Media Conversions
     **************************************************************************/
    public function registerMediaConversions(Media $media = null)
    {
        $this->addMediaConversion('medium')
            ->width(1024)
            ->height(1024);

        $this->addMediaConversion('thumb')
            ->width(256)
            ->height(256);
    }

    /***************************************************************************
     * Expiration Interval
     **************************************************************************/
    public static function getExpirationInterval()
    {
        return CarbonInterval::months(1);
    }

    /***************************************************************************
     * Type Values
     **************************************************************************/
    const TYPE_LOST = 'lost';
    const TYPE_FOUND = 'found';

    const TYPES = [
        self::TYPE_LOST,
        self::TYPE_FOUND,
    ];

    /***************************************************************************
     * State Values
     **************************************************************************/
    const STATE_WAITING = 'waiting';
    const STATE_SUSPENDED = 'suspended';
    const STATE_DONE = 'done';

    const STATES = [
        self::STATE_WAITING,
        self::STATE_SUSPENDED,
        self::STATE_DONE,
    ];

    /***************************************************************************
     * Deletion Reason Values
     **************************************************************************/
    const DELETION_REASON_DEAD = 'dead';
    const DELETION_REASON_FOUND = 'found';
    const DELETION_REASON_MISTAKE = 'mistake';
    const DELETION_REASON_SECURE = 'secure';
    const DELETION_REASON_OTHER = 'other';

    const DELETION_REASONS = [
        self::DELETION_REASON_FOUND,
        self::DELETION_REASON_SECURE,
        self::DELETION_REASON_DEAD,
        self::DELETION_REASON_MISTAKE,
        self::DELETION_REASON_OTHER,
    ];

    // End of class
}
