<?php

namespace Modules\Blog\Entities;

use Illuminate\Database\Eloquent\Model;

use Backpack\CRUD\app\Models\Traits\CrudTrait;
use Spatie\MediaLibrary\Models\Media;
use Spatie\MediaLibrary\HasMedia\HasMedia;
use Spatie\MediaLibrary\HasMedia\HasMediaTrait;

use Modules\Blog\Entities\Presenters\PostPresenter;

use Str;

class Post extends Model implements HasMedia
{
    use PostPresenter, HasMediaTrait, CrudTrait;
    
    /***************************************************************************
     * Attributes
     **************************************************************************/
    protected $fillable = [
        'title', 'slug', 'excerpt', 'body', 'image', 'state',
        'meta_title', 'meta_description',
    ];
    
    /***************************************************************************
     * Boot
     **************************************************************************/
     protected static function boot() {
        parent::boot();

        static::creating(function ($post) {
            $post->slug = $post->slug ?? Str::slug($post->title);
        });
    }
    
    /***************************************************************************
     * Getters
     **************************************************************************/
    public function getImageAttribute()
    {
        return $this->getFirstMediaUrl();
    }
        
    /***************************************************************************
     * Setters
     **************************************************************************/
    public function setImageAttribute($value)
    {
        if($value) {
            if ($this->hasMedia() && $value == $this->getMedia()->first()->getFullUrl()) {
                return; // This is the same image link
            }
            
            $this->addMediaFromBase64($value)
                    ->usingFileName(Str::random(60).'.jpg')
                    ->toMediaCollection();      
        }
    }
    
    /***************************************************************************
     * Scopes
     **************************************************************************/
    public function scopePublished($query)
    {
        return $query->where('state', Post::STATE_PUBLISHED);
    }
    
    /***************************************************************************
     * Relationships
     **************************************************************************/
    public function tags()
    {
        return $this->belongsToMany(Tag::class);
    }
    
    /***************************************************************************
     * Media
     **************************************************************************/
    public function registerMediaCollections()
    {
        $this->addMediaCollection('default')->onlyKeepLatest(1);
    }
    
    public function registerMediaConversions(Media $media = null)
    {
        $this->addMediaConversion('thumb')
            ->width(512)
            ->height(512);
    }
    
    /***************************************************************************
     * State Values
     **************************************************************************/
    const STATE_PENDING = 'pending';
    const STATE_PUBLISHED = 'published'; 
    
    const STATES = [
        self::STATE_PENDING,
        self::STATE_PUBLISHED,
    ];
    
    // End of class
}
