<?php

namespace Modules\Api\Tests\Feature;

use Illuminate\Http\UploadedFile;

use Modules\Common\Entities\Animal;
use Modules\Common\Entities\Report;
use Modules\Common\Entities\User;


class ReportTest extends TestCase
{

    /***************************************************************************
     * Get Reports
     **************************************************************************/
    public function testGetReportsWithUnauthenticated()
    {
        $this->getJson('/api/reports')
            ->assertStatus(401)
            ->assertJson(['message' => 'Unauthenticated.']);
    }

    public function testGetReports()
    {
        $user = User::find(1);

        $this->actingAs($user, 'api')
            ->getJson('/api/reports')
            ->assertStatus(200);
    }

    /***************************************************************************
     * Get Report
     **************************************************************************/
    public function testGetReportWithUnauthenticated()
    {
        $this->getJson('/api/reports/1')
            ->assertStatus(401)
            ->assertJson(['message' => 'Unauthenticated.']);
    }

    public function testGetReportSuccessfully()
    {
        $user = User::find(1);

        $this->actingAs($user, 'api')
            ->getJson('/api/reports/1')
            ->assertStatus(200);
    }

    /***************************************************************************
     * Store Reports
     **************************************************************************/
    public function testStoreReportWithUnauthenticated()
    {
        $this->postJson('/api/reports')
            ->assertStatus(401)
            ->assertJson(['message' => 'Unauthenticated.']);
    }

    public function testStoreReportSuccessfully()
    {
        $user = User::find(1);

        $payload = [
            'location_address' => 'Here',
            'location_latitude' => 4,
            'location_longitude' => 50,
            'type' => Report::TYPE_FOUND,
            'animal_species' => Animal::ANIMAL_SPECIES_PET_DOG_SMALL,
            'animal_condition' => Animal::ANIMAL_CONDITION_DEAD,
            'comment' => 'Comment',
            'state' => Report::STATE_WAITING,
            'images' => [
                UploadedFile::fake()->image('image1.jpg'),
                UploadedFile::fake()->image('image2.jpg'),
            ],
        ];


        $this->actingAs($user, 'api')
            ->postJson('/api/reports', $payload)
            ->assertStatus(201)
            ->assertJsonStructure([
                'data' => [
                    'id',
                    'type',
                    'animal_species',
                    'animal_condition',
                    'identification_type',
                    'identification_number',
                    'comment',
                    'state',
                    'created_at',
                    'expires_at',
                    'location' => [
                        'address',
                        'latitude',
                        'longitude',
                    ],
                    'images' => [],
                    'thumbnail',
                    'user' => [
                        'id',
                        'first_name',
                        'last_name',
                        'email',
                        'phone',
                        'phone_is_private',
                        'address',
                    ],
                ],
            ]);
    }

    /***************************************************************************
     * Update Report
     **************************************************************************/
    public function testUpdateReportWithUnauthenticated()
    {
        $this->putJson('/api/reports/1')
            ->assertStatus(401)
            ->assertJson(['message' => 'Unauthenticated.']);
    }

    public function testUpdateReportWithForbidden()
    {
        $user = User::find(3);

        $payload = [
            'location_latitude' => 4,
            'location_longitude' => 50,
            'type' => 'found',
            'animal_species' => Animal::ANIMAL_SPECIES_PET_DOG_SMALL,
            'animal_condition' => 'dead',
            'comment' => 'Comment',
            'state' => 'waiting',
        ];

        $this->actingAs($user, 'api')
            ->putJson('/api/reports/1', $payload)
            ->assertStatus(403)
            ->assertJson(['message' => 'This action is unauthorized.']);
    }

    public function testUpdateReportSuccessfully()
    {
        $user = User::find(1);

        $payload = [
            'location_latitude' => 4,
            'location_longitude' => 50,
            'type' => 'found',
            'animal_species' => Animal::ANIMAL_SPECIES_PET_DOG_SMALL,
            'animal_condition' => 'dead',
            'comment' => 'Comment',
            'state' => 'waiting',
        ];

        $this->actingAs($user, 'api')
            ->putJson('/api/reports/1', $payload)
            ->assertStatus(200)
            ->assertJsonStructure([
                'data' => [
                    'id',
                    'type',
                    'animal_species',
                    'animal_condition',
                    'identification_type',
                    'identification_number',
                    'comment',
                    'state',
                    'created_at',
                    'expires_at',
                    'location' => [
                        'address',
                        'latitude',
                        'longitude',
                    ],
                    'images' => [],
                    'thumbnail',
                    'user' => [
                        'id',
                        'first_name',
                        'last_name',
                        'email',
                        'phone',
                        'phone_is_private',
                        'address',
                    ],
                ],
            ]);
    }

    /***************************************************************************
     * Delete Report
     **************************************************************************/
    public function testDeleteReportWithUnauthenticated()
    {
        $this->deleteJson('/api/reports/1')
            ->assertStatus(401)
            ->assertJson(['message' => 'Unauthenticated.']);
    }

    public function testDeleteReportWithForbidden()
    {
        $user = User::find(3);

        $payload = [
            'deletion_reason' => Report::DELETION_REASON_OTHER,
            'deletion_reason_comment' => 'Comment',
        ];

        $this->actingAs($user, 'api')
            ->deleteJson('/api/reports/1', $payload)
            ->assertStatus(403)
            ->assertJson(['message' => 'This action is unauthorized.']);
    }

    public function testDeleteReportSuccessfully()
    {
        $user = User::find(1);

        $payload = [
            'deletion_reason' => Report::DELETION_REASON_OTHER,
            'deletion_reason_comment' => 'Comment',
        ];

        $this->actingAs($user, 'api')
            ->deleteJson('/api/reports/1', $payload)
            ->assertStatus(204);
    }

    public function testDeleteReportWithReason()
    {
        $user = User::find(1);

        $payload = [
            'deletion_reason' => Report::DELETION_REASON_OTHER,
            'deletion_reason_comment' => 'Comment',
        ];

        $this->actingAs($user, 'api')
            ->deleteJson('/api/reports/1', $payload)
            ->assertStatus(204);

        $report = Report::withTrashed()->find(1);
        $this->assertTrue($report->deletion_reason == Report::DELETION_REASON_OTHER);
        $this->assertTrue($report->deletion_reason_comment == $report->deletion_reason_comment);
    }

    /***************************************************************************
     * Extend Expiration
     **************************************************************************/
    public function testExtendExpirationReport()
    {
        $user = User::find(1);

        $this->actingAs($user, 'api')
            ->getJson('/api/reports/1/extend_expiration')
            ->assertStatus(200)
            ->assertJsonStructure([
                'data' => [
                    'id',
                    'type',
                    'animal_species',
                    'animal_condition',
                    'identification_type',
                    'identification_number',
                    //'comment',
                    'state',
                    'created_at',
                    'expires_at',
                    'location' => [
                        'address',
                        'latitude',
                        'longitude',
                    ],
                    'images' => [],
                    'thumbnail',
                    'user' => [
                        'id',
                        'first_name',
                        'last_name',
                        'email',
                        'phone',
                        'phone_is_private',
                        'address',
                    ],
                ],
            ]);
    }

    // End of class
}
