<?php

namespace Modules\Api\Tests\Feature;

use Illuminate\Http\UploadedFile;

use Modules\Common\Entities\Animal;
use Modules\Common\Entities\User;


class AlertTest extends TestCase
{

    /***************************************************************************
     * Test Store Alert
     **************************************************************************/
    public function testStoreAlertWithUnauthenticated()
    {
        $this->postJson('/api/alerts')
            ->assertStatus(401)
            ->assertJson(['message' => 'Unauthenticated.']);
    }

    public function testStoreAlertWithNoPremium()
    {
        $user = User::find(3);

        $this->actingAs($user, 'api')
            ->postJson('/api/alerts')
            ->assertStatus(403)
            ->assertJson(['error' => 'account_type_not_allowed']);
    }

    public function testStoreAlertWithOrganization()
    {
        $user = User::find(1);
        $user->account_type = User::ACCOUNT_TYPE_ORGANIZATION;
        $user->save();

        $payload = [
            'location_address' => 'Here',
            'location_latitude' => 4,
            'location_longitude' => 50,
            'animal_species' => Animal::ANIMAL_SPECIES_PET_DOG_SMALL,
            'radius' => 25,
        ];

        $this->actingAs($user, 'api')
            ->postJson('/api/alerts', $payload)
            ->assertStatus(201);
    }

    public function testStoreAlertWithOrganizationAccess()
    {
        $user = User::find(1);
        $user->account_type = User::ACCOUNT_TYPE_ACCESS;
        $user->save();

        $payload = [
            'location_address' => 'Here',
            'location_latitude' => 4,
            'location_longitude' => 50,
            'animal_species' => Animal::ANIMAL_SPECIES_PET_DOG_SMALL,
            'radius' => 25,
        ];

        $this->actingAs($user, 'api')
            ->postJson('/api/alerts', $payload)
            ->assertStatus(201);
    }

    public function testStoreAlertSuccessfully()
    {
        $user = User::find(1);

        $payload = [
            'location_address' => 'Here',
            'location_latitude' => 4,
            'location_longitude' => 50,
            'animal_species' => Animal::ANIMAL_SPECIES_PET_DOG_SMALL,
            'radius' => 25,
        ];

        $this->actingAs($user, 'api')
            ->postJson('/api/alerts', $payload)
            ->assertStatus(201)
            ->assertJson([
                'data' => [
                    'id' => 6,
                    'animal_species' => Animal::ANIMAL_SPECIES_PET_DOG_SMALL,
                    'location' => [
                        'address' => 'Here',
                        'longitude' => 50,
                        'latitude' => 4,
                        'zipcode' => '',
                        'country' => '',
                    ],
                    'radius' => 25,
                ],
            ]);
    }

    /***************************************************************************
     * Test Update Alert
     **************************************************************************/
    public function testUpdateAlertWithUnauthenticated()
    {
        $this->putJson('/api/alerts/2')
            ->assertStatus(401)
            ->assertJson(['message' => 'Unauthenticated.']);
    }

    public function testUpdateAlertWithForbidden()
    {
        $user = User::find(1);

        $payload = [
            'location_address' => 'Here',
            'location_latitude' => 4,
            'location_longitude' => 50,
            'animal_species' => Animal::ANIMAL_SPECIES_PET_DOG_SMALL,
            'radius' => 25,
        ];

        $this->actingAs($user, 'api')
            ->putJson('/api/alerts/2', $payload)
            ->assertStatus(403)
            ->assertJson(['message' => 'This action is unauthorized.']);
    }

    public function testUpdateAlertWithNoPremium()
    {
        $user = User::find(3);

        $this->actingAs($user, 'api')
            ->putJson('/api/alerts/2')
            ->assertStatus(403)
            ->assertJson(['error' => 'account_type_not_allowed']);
    }

    public function testUpdateAlertWithOrganization()
    {
        $user = User::find(1);
        $user->account_type = User::ACCOUNT_TYPE_ORGANIZATION;
        $user->save();

        $payload = [
            'location_address' => 'Here',
            'location_latitude' => 4,
            'location_longitude' => 50,
            'animal_species' => Animal::ANIMAL_SPECIES_PET_DOG_SMALL,
            'radius' => 25,
        ];

        $this->actingAs($user, 'api')
            ->putJson('/api/alerts/1', $payload)
            ->assertStatus(200);
    }

    public function testUpdateAlertWithOrganizationAccess()
    {
        $user = User::find(1);
        $user->account_type = User::ACCOUNT_TYPE_ACCESS;
        $user->save();

        $payload = [
            'location_address' => 'Here',
            'location_latitude' => 4,
            'location_longitude' => 50,
            'animal_species' => Animal::ANIMAL_SPECIES_PET_DOG_SMALL,
            'radius' => 25,
        ];

        $this->actingAs($user, 'api')
            ->putJson('/api/alerts/1', $payload)
            ->assertStatus(200);
    }

    public function testUpdateAlertSuccessfully()
    {
        $user = User::find(1);

        $payload = [
            'location_address' => 'Here',
            'location_latitude' => 4,
            'location_longitude' => 50,
            'animal_species' => Animal::ANIMAL_SPECIES_PET_DOG_SMALL,
            'radius' => 25,
        ];

        $this->actingAs($user, 'api')
            ->putJson('/api/alerts/1', $payload)
            ->assertStatus(200)
            ->assertJson([
                'data' => [
                    'id' => 1,
                    'animal_species' => Animal::ANIMAL_SPECIES_PET_DOG_SMALL,
                    'location' => [
                        'address' => 'Here',
                        'longitude' => 50,
                        'latitude' => 4,
                        'zipcode' => '',
                        'country' => '',
                    ],
                    'radius' => 25,
                ],
            ]);
    }

    /***************************************************************************
     * Test Delete Alert
     **************************************************************************/
    public function testDeleteAlertWithUnauthenticated()
    {
        $this->deleteJson('/api/alerts/2')
            ->assertStatus(401)
            ->assertJson(['message' => 'Unauthenticated.']);
    }

    public function testDeleteAlertWithForbidden()
    {
        $user = User::find(1);

        $this->actingAs($user, 'api')
            ->deleteJson('/api/alerts/2')
            ->assertStatus(403)
            ->assertJson(['message' => 'This action is unauthorized.']);
    }

    public function testDeleteAlertWithNoPremium()
    {
        $user = User::find(3);

        $this->actingAs($user, 'api')
            ->deleteJson('/api/alerts/2')
            ->assertStatus(403)
            ->assertJson(['error' => 'account_type_not_allowed']);
    }

    public function testDeleteAlertWithOrganization()
    {
        $user = User::find(1);
        $user->account_type = User::ACCOUNT_TYPE_ORGANIZATION;
        $user->save();

        $this->actingAs($user, 'api')
            ->deleteJson('/api/alerts/1')
            ->assertStatus(204);
    }

    public function testDeleteAlertWithOrganizationAccess()
    {
        $user = User::find(1);
        $user->account_type = User::ACCOUNT_TYPE_ACCESS;
        $user->save();

        $this->actingAs($user, 'api')
            ->deleteJson('/api/alerts/1')
            ->assertStatus(204);
    }

    public function testDeleteAlertSuccessfully()
    {
        $user = User::find(1);

        $this->actingAs($user, 'api')
            ->deleteJson('/api/alerts/1')
            ->assertStatus(204);
    }

    // End of class
}
