<?php

namespace Modules\Api\Http\Services;

use Modules\Common\Entities\User;

use Auth;
use Hash;
use Log;

class AuthService
{
    /***************************************************************************
     * Methods
     **************************************************************************/
    public function register(array $attributes, $autovalidate = false)
    {
        // Hash password
        $attributes['password'] = Hash::make($attributes['password']);

        // Create user
        $user = User::create($attributes);

        // Set autovalidation
        if ($autovalidate) {
            $user->markEmailAsVerified();
        } else {
            $this->sendEmailVerificationNotification($user);
        }

        // Return user
        return $user;
    }

    public function login(array $attributes)
    {
        // Authenticate user
        if (!Auth::guard('user')->attempt($attributes)) {
            abort(401, __('auth.failed'));
        }

        // Get user
        $user = Auth::guard('user')->user();

        // Check if the user is verified
        if (!$user->hasVerifiedEmail()) {
            $this->sendEmailVerificationNotification($user);
            abort(403, __('auth.not_verified'));
        }

        // Check if the user is blocked
        if ($user->is_blocked) {
            abort(403, __('auth.account_is_blocked'));
        }

        // Create token
        $tokenResult = $user->generateAccessToken();

        // Return token
        return $tokenResult;
    }

    public function logout()
    {
        // Delete token
        Auth::user()->token()->delete();
    }

    /***************************************************************************
     * Utils
     **************************************************************************/
    private function sendEmailVerificationNotification(User $user)
    {
        try {
            $user->sendEmailVerificationNotification();
        } catch (\Exception $e) {
            Log::error('Email verification to user ' . $user->email . ' not sent');
            Log::debug($e->getMessage());
        }
    }

    // End of class
}
