<?php

namespace Modules\Api\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Str;
use App\Http\Controllers\Controller;
use Modules\Api\Http\Resources\Report as ReportResource;
use Modules\Api\Http\Resources\Reports as ReportCollection;
use Modules\Api\Http\Services\ReportService;
use Modules\Common\Entities\Report;
use Modules\Common\Http\Requests\ReportDeleteRequest;
use Modules\Common\Http\Requests\ReportStoreRequest;
use Modules\Common\Http\Requests\ReportUpdateRequest;

class ReportController extends Controller
{

    /***************************************************************************
     * Attributes
     **************************************************************************/
    protected $reportService;

    /***************************************************************************
     * Constructors
     **************************************************************************/
    public function __construct(ReportService $reportService)
    {
        $this->reportService = $reportService;
    }

    /***************************************************************************
     * Methods
     **************************************************************************/
    public function index(Request $request)
    {
        // Get reports
        $reports = $this->reportService->index($request->all());

        // Return reports
        return new ReportCollection($reports);
    }

    public function show(Report $report)
    {
        // Return resource
        return new ReportResource($report);
    }

    public function store(ReportStoreRequest $request)
    {
        // Create resource
        $report = $this->reportService->store($request->validated());

        // Handle images
        $this->handleImages($request, $report);

        // Refresh resource
        $report->refresh();

        // Handle notifies
        $report->handleNotifies(true);

        // Return response
        return new ReportResource($report);
    }

    public function update(ReportUpdateRequest $request, Report $report)
    {
        // Authorize user
        $this->authorize('write', $report);

        // Update resource
        $report->update($request->validated());

        // Refresh resource
        $report->refresh();

        // Handle notifies
        $report->handleNotifies();

        // Return response
        return new ReportResource($report);
    }

    public function delete(ReportDeleteRequest $request, Report $report)
    {
        // Authorize user
        $this->authorize('write', $report);

        // Delete resource
        $this->reportService->delete($request->validated(), $report);

        // Return response
        return response()->noContent();
    }

    public function extendExpiration(Report $report)
    {
        // Authorize user
        $this->authorize('write', $report);

        // Update resource
        $report->extendExpiration();

        // Refresh resource
        $report->refresh();

        // Return response
        return new ReportResource($report);
    }

    public function suspend(Report $report)
    {
        // Authorize user
        $this->authorize('write', $report);

        // Update resource
        $report->suspend();

        // Refresh resource
        $report->refresh();

        // Return response
        return new ReportResource($report);
    }

    public function unsuspend(Report $report)
    {
        // Authorize user
        $this->authorize('write', $report);

        // Update resource
        $report->unsuspend();

        // Refresh resource
        $report->refresh();

        // Return response
        return new ReportResource($report);
    }

    /***************************************************************************
     * Utils
     **************************************************************************/
    private function handleImages(Request $request, Report $report)
    {
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $image) {
                $report->addMedia($image)
                    ->usingFileName(Str::random(60) . '.jpg')
                    ->toMediaCollection();
            }
        }
    }

    // End of class
}
