<?php

namespace Modules\Admin\Http\Controllers;

use Backpack\CRUD\app\Http\Controllers\CrudController;
use Backpack\CRUD\app\Library\CrudPanel\CrudPanelFacade as CRUD;

use Modules\Admin\Http\Requests\UpdateUserRequest;
use Modules\Admin\Http\Controllers\Traits\RestoreOperation;
use Modules\Admin\Http\Controllers\Traits\ResendVerificationEmailOperation;
use Modules\Common\Entities\User;

use Hash;


class UserCrudController extends CrudController
{
    use \Backpack\CRUD\app\Http\Controllers\Operations\ListOperation;
    use \Backpack\CRUD\app\Http\Controllers\Operations\ShowOperation {
        show as traitShow;
    }
    use \Backpack\CRUD\app\Http\Controllers\Operations\UpdateOperation {
        update as traitUpdate;
    }
    use \Backpack\CRUD\app\Http\Controllers\Operations\DeleteOperation;
    use RestoreOperation;
    use ResendVerificationEmailOperation;

    /***************************************************************************
     * Setup
     **************************************************************************/
    public function setup()
    {
        CRUD::setModel("Modules\Common\Entities\User");
        CRUD::setRoute(config('backpack.base.route_prefix') . '/users');
        CRUD::setEntityNameStrings(__('messages.global.user'), __('messages.global.users'));
        CRUD::set('show.setFromDb', false);
    }

    /***************************************************************************
     * List
     **************************************************************************/
    public function setupListOperation()
    {
        CRUD::addClause('isIndividual');

        CRUD::addColumns($this->fields());
        CRUD::removeColumns([
            'password', 'phone', 'phone_is_private', 'address', 'gender',
            'date_of_birth', 'email_verified_at', 'phone_verified_at',
            'deleted_at', 'force_premium', 'is_blocked',
        ]);
        CRUD::addColumn([
            'name' => 'has_verified_email',
            'type' => 'boolean',
            'label' => __('messages.global.has_verified_email'),
            'wrapper' => [
                'element' => 'span',
                'class' => function ($crud, $column, $entry, $related_key) {
                    return $related_key ? 'badge badge-success' : 'badge badge-warning';
                },
            ],
            'orderable'  => true,
            'orderLogic' => function ($query, $column, $columnDirection) {
                return $query->orderBy('email_verified_at', $columnDirection);
            }
        ]);
        CRUD::addColumn([
            'name' => 'has_verified_phone',
            'type' => 'boolean',
            'label' => __('messages.global.has_verified_phone'),
            'wrapper' => [
                'element' => 'span',
                'class' => function ($crud, $column, $entry, $related_key) {
                    return $related_key ? 'badge badge-success' : 'badge badge-warning';
                },
            ],
            'orderable'  => true,
            'orderLogic' => function ($query, $column, $columnDirection) {
                return $query->orderBy('phone_verified_at', $columnDirection);
            }
        ]);
        CRUD::addColumn([
            'name' => 'is_premium',
            'type' => 'boolean',
            'label' => __('messages.global.is_premium'),
            'wrapper' => [
                'element' => 'span',
                'class' => function ($crud, $column, $entry, $related_key) {
                    return $related_key ? 'badge badge-success' : 'badge badge-warning';
                },
            ],
        ]);
        CRUD::addColumn([
            'name' => 'is_blocked',
            'type' => 'boolean',
            'label' => __('messages.global.is_blocked'),
            'wrapper' => [
                'element' => 'span',
                'class' => function ($crud, $column, $entry, $related_key) {
                    return $related_key ? 'badge badge-danger' : 'badge badge-success';
                },
            ],
            'orderable'  => true,
            'orderLogic' => function ($query, $column, $columnDirection) {
                return $query->orderBy('is_blocked', $columnDirection);
            }
        ]);
        CRUD::addFilter(
            [
                'type'  => 'text',
                'name'  => 'location_zipcode',
                'label' => __('messages.global.zipcode'),
            ],
            false,
            function ($value) {
                CRUD::addClause('where', 'location_zipcode', 'like', "$value");
            }
        );
        CRUD::addFilter(
            [
                'type'  => 'dropdown',
                'name'  => 'is_premium',
                'label' => __('messages.global.is_premium'),
            ],
            [
                1 => 'Oui',
                2 => 'Non',
            ],
            function ($value) {
                CRUD::addClause($value == 1 ? 'filterByPremium' : 'filterByNotPremium');
            }
        );

        $this->addRestoreFilter();
    }

    /***************************************************************************
     * Show
     **************************************************************************/
    public function setupShowOperation()
    {
        CRUD::addButtonFromView('line', 'userReports', 'user_reports', 'beginning');
        CRUD::addButtonFromView('line', 'resendVerificationEmail', 'resend_verification_email');

        CRUD::addColumns($this->fields());
        CRUD::removeColumns(['password', 'force_premium', 'is_blocked']);
        CRUD::addColumn([
            'name' => 'has_verified_email',
            'type' => 'boolean',
            'label' => __('messages.global.has_verified_email'),
            'wrapper' => [
                'element' => 'span',
                'class' => function ($crud, $column, $entry, $related_key) {
                    return $related_key ? 'badge badge-success' : 'badge badge-warning';
                },
            ],
        ]);
        CRUD::addColumn([
            'name' => 'has_verified_phone',
            'type' => 'boolean',
            'label' => __('messages.global.has_verified_phone'),
            'wrapper' => [
                'element' => 'span',
                'class' => function ($crud, $column, $entry, $related_key) {
                    return $related_key ? 'badge badge-success' : 'badge badge-warning';
                },
            ],
        ]);
        CRUD::addColumn([
            'name' => 'is_premium',
            'type' => 'boolean',
            'label' => __('messages.global.is_premium'),
            'wrapper' => [
                'element' => 'span',
                'class' => function ($crud, $column, $entry, $related_key) {
                    return $related_key ? 'badge badge-success' : 'badge badge-warning';
                },
            ],
        ]);
        CRUD::addColumn([
            'name' => 'is_blocked',
            'type' => 'boolean',
            'label' => __('messages.global.is_blocked'),
            'wrapper' => [
                'element' => 'span',
                'class' => function ($crud, $column, $entry, $related_key) {
                    return $related_key ? 'badge badge-danger' : 'badge badge-success';
                },
            ],
        ]);
    }

    public function show($id)
    {
        $this->crud->entry = $this->crud->model->withTrashed()->findOrFail($id);
        $this->crud->entry = $this->crud->entry->withFakes();

        return $this->traitShow($id);
    }


    /***************************************************************************
     * Update
     **************************************************************************/
    public function setupUpdateOperation()
    {
        CRUD::setValidation(UpdateUserRequest::class);
        CRUD::addFields($this->fields());
        CRUD::removeFields([
            'created_at', 'updated_at',
        ]);
    }

    public function update()
    {
        CRUD::setRequest(CRUD::validateRequest());
        CRUD::setRequest($this->handlePasswordInput(CRUD::getRequest()));
        CRUD::unsetValidation(); // validation has already been run

        return $this->traitUpdate();
    }

    /***************************************************************************
     * Utils
     **************************************************************************/
    private function handlePasswordInput($request)
    {
        // Encrypt password if specified.
        if ($request->input('password')) {
            $request->request->set('password', Hash::make($request->input('password')));
        } else {
            $request->request->remove('password');
        }

        return $request;
    }

    /***************************************************************************
     * Fields
     **************************************************************************/
    private function fields()
    {
        return [
            [
                'name' => 'email',
                'type' => 'email',
                'label' => __('messages.global.email_address'),
            ],
            [
                'name' => 'password',
                'type' => 'password',
                'label' => __('messages.global.password')
            ],
            [
                'name' => 'first_name',
                'type' => 'text',
                'label' => __('messages.global.first_name')
            ],
            [
                'name' => 'last_name',
                'type' => 'text',
                'label' => __('messages.global.last_name')
            ],
            [
                'name' => 'gender',
                'type' => 'select_from_array',
                'label' => __('messages.global.gender'),
                'allows_null' => true,
                'options' => [
                    User::GENDER_MALE => __('messages.gender.' . User::GENDER_MALE),
                    User::GENDER_FEMALE => __('messages.gender.' . User::GENDER_FEMALE),
                    User::GENDER_UNSPECIFIED => __('messages.gender.' . User::GENDER_UNSPECIFIED),
                ],
            ],
            [
                'name' => 'date_of_birth',
                'type' => 'text',
                'label' => __('messages.global.date_of_birth')
            ],
            [
                'name' => 'phone',
                'type' => 'text',
                'label' => __('messages.global.phone_number')
            ],
            [
                'name' => 'phone_is_private',
                'type' => 'boolean',
                'label' => __('messages.global.hide_phone_number')
            ],
            [
                'name' => 'address',
                'type' => 'text',
                'label' => __('messages.global.address')
            ],
            [
                'name' => 'location_zipcode',
                'type' => 'text',
                'label' => __('messages.global.zipcode')
            ],
            [
                'name' => 'location_country',
                'type' => 'text',
                'label' => __('messages.global.country')
            ],
            [
                'name' => 'email_verified_at',
                'type' => 'datetime',
                'label' => __('messages.global.email_verified_at'),
            ],
            [
                'name' => 'phone_verified_at',
                'type' => 'datetime',
                'label' => __('messages.global.phone_verified_at'),
            ],
            [
                'name' => 'created_at',
                'type' => 'datetime',
                'label' => __('messages.global.created_at'),
            ],
            [
                'name' => 'deleted_at',
                'type' => 'datetime',
                'label' => __('messages.global.deleted_at'),
            ],
            [
                'name' => 'force_premium',
                'type' => 'boolean',
                'label' => __('messages.global.force_premium')
            ],
            [
                'name' => 'is_blocked',
                'type' => 'boolean',
                'label' => __('messages.global.is_blocked')
            ],
        ];
    }

    // End of class
}
