<?php

namespace Modules\User\Http\Requests;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;
use Modules\Common\Entities\User;


class RegisterRequest extends FormRequest
{

    public function authorize()
    {
        return true;
    }

    public function rules()
    {
        return [
            'account_type'          => ['nullable', Rule::in([User::ACCOUNT_TYPE_INDIVIDUAL, User::ACCOUNT_TYPE_ORGANIZATION])],
            'email'                 => ['required', 'email:rfc,dns', 'max:180', 'unique:users,email,' . $this->id . ',id,account_type,' . $this->account_type],
            'password'              => ['required', 'string', 'min:8'],
            'first_name'            => ['required', 'regex:/^[\pL\s\-]+$/u', 'max:30'],
            'last_name'             => ['required', 'regex:/^[\pL\s\-]+$/u', 'max:30'],
            'address'               => ['required', 'string', 'max:180'],
            'location_latitude'     => ['required', 'numeric', 'min:-90',  'max:90'],
            'location_longitude'    => ['required', 'numeric', 'min:-180', 'max:180'],
            'location_zipcode'      => ['nullable', 'string', 'max:20'],
            'location_country'      => ['nullable', 'string', 'max:30'],
            'gender'                => ['nullable', Rule::in(User::GENDERS)],
            'date_of_birth'         => ['nullable', 'date_format:Y-m-d'],
            'phone'                 => ['required', 'phone:AUTO', 'max:20'],
            'opt_in'                => ['required'],
            'organization_function' => ['exclude_if:account_type,' . User::ACCOUNT_TYPE_INDIVIDUAL, 'required', 'string', 'max:30'],
            'organization_name'     => ['exclude_if:account_type,' . User::ACCOUNT_TYPE_INDIVIDUAL, 'required', 'string', 'max:30'],
            'organization_number'   => ['exclude_if:account_type,' . User::ACCOUNT_TYPE_INDIVIDUAL, 'required', 'string', 'max:30'],
        ];
    }

    protected function getValidatorInstance()
    {
        $data = $this->all();

        // Set default value to account_type if null
        if (!isset($data['account_type'])) {
            $data['account_type'] = User::ACCOUNT_TYPE_INDIVIDUAL;
            $this->getInputSource()->replace($data);
        }

        return parent::getValidatorInstance();
    }

    public function attributes()
    {
        return [
            "address" => "adresse",
            "location_latitude" => "latitude",
            "location_longitude" => "longitude",
            "location_zipcode" => "code postal",
            "location_country" => "pays",
            "organization_function" => "fonction dans l'organisation",
            "organization_name" => "nom de l'organisation",
            "organization_number" => "numéro d'entreprise",
            "opt_in" => "conditions générales d'utilisation",
        ];
    }

    // End of class
}
