<?php

namespace Modules\User\Http\Controllers\Auth;

use Illuminate\Auth\Access\AuthorizationException;
use Illuminate\Auth\Events\Verified;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Foundation\Auth\VerifiesEmails;
use App\Http\Controllers\Controller;
use Modules\Common\Entities\User;

class VerificationController extends Controller
{
    /*
      |-------------------------------------------------------------------------
      | Email Verification Controller
      |-------------------------------------------------------------------------
      |
      | This controller is responsible for handling email verification for any
      | user that recently registered with the application. Emails may also
      | be re-sent if the user didn't receive the original email message.
      |
     */

    use VerifiesEmails;

    public function __construct()
    {
        $this->middleware('auth:user')->except('verify');
        $this->middleware('throttle:6,1')->only('verify', 'resend');
    }

    protected function redirectTo()
    {
        return route('user.profile.index');
    }

    public function show(Request $request)
    {
        return $request->user()->hasVerifiedEmail()
            ? redirect($this->redirectPath())
            : view('user::auth.verify');
    }

    public function verify($userId, Request $request)
    {
        $user = User::findOrFail($userId);

        if ($user->hasVerifiedEmail()) {
            throw new AuthorizationException(trans('auth.verification_already_verified'));
        }

        if (!$request->hasValidSignature()) {
            $user->sendEmailVerificationNotification();
            throw new AuthorizationException(trans('auth.verification_expired'));
        }

        if ($user->markEmailAsVerified()) {
            event(new Verified($user));
        }

        if ($response = $this->verified($request)) {
            return $response;
        }

        auth('user')->login($user);

        return $request->wantsJson()
            ? new Response('', 204)
            : redirect($this->redirectPath())->with('verified', true);
    }

    // End of class
}
