<?php

namespace Modules\Api\Tests\Feature;

use Illuminate\Http\UploadedFile;

use Modules\Common\Entities\Animal;
use Modules\Common\Entities\User;


class RegisteredAnimalTest extends TestCase
{

    /***************************************************************************
     * Test Get Registered Animal
     **************************************************************************/
    public function testGetRegisteredAnimalWithUnauthenticated()
    {
        $this->getJson('/api/registered_animals/2')
            ->assertStatus(401)
            ->assertJson(['message' => 'Unauthenticated.']);
    }

    public function testGetRegisteredAnimalWithForbidden()
    {
        $user = User::find(1);

        $this->actingAs($user, 'api')
            ->getJson('/api/registered_animals/2')
            ->assertStatus(403)
            ->assertJson(['message' => 'This action is unauthorized.']);
    }

    public function testGetRegisteredAnimalWithNoPremium()
    {
        $user = User::find(3);

        $this->actingAs($user, 'api')
            ->getJson('/api/registered_animals/2')
            ->assertStatus(403)
            ->assertJson(['error' => 'account_not_premium']);
    }

    public function testGetRegisteredAnimalWithOrganization()
    {
        $user = User::find(1);
        $user->account_type = User::ACCOUNT_TYPE_ORGANIZATION;
        $user->save();

        $this->actingAs($user, 'api')
            ->getJson('/api/registered_animals/2')
            ->assertStatus(403)
            ->assertJson(['error' => 'account_type_not_allowed']);
    }

    public function testGetRegisteredAnimalWithOrganizationAccess()
    {
        $user = User::find(1);
        $user->account_type = User::ACCOUNT_TYPE_ACCESS;
        $user->save();

        $this->actingAs($user, 'api')
            ->getJson('/api/registered_animals/2')
            ->assertStatus(403)
            ->assertJson(['error' => 'account_type_not_allowed']);
    }

    public function testGetRegisteredAnimalSuccessfully()
    {
        $user = User::find(1);

        $this->actingAs($user, 'api')
            ->getJson('/api/registered_animals/1')
            ->assertStatus(200)
            ->assertJson([
                'data' =>  [
                    'id' => 1,
                    'name' => 'Medor',
                    'animal_species' => Animal::ANIMAL_SPECIES_PET_DOG_COCKER,
                    'identification_type' => Animal::IDENTIFICATION_TYPE_MICROCHIP,
                    'identification_number' => '1122',
                    'comment' => 'Comment',
                    'images' => [],
                    'thumbnail' => '',
                ],
            ]);
    }

    /***************************************************************************
     * Test Store Registered Animal
     **************************************************************************/
    public function testStoreRegisteredAnimalWithUnauthenticated()
    {
        $this->postJson('/api/registered_animals')
            ->assertStatus(401)
            ->assertJson(['message' => 'Unauthenticated.']);
    }

    public function testStoreRegisteredAnimalWithNoPremium()
    {
        $user = User::find(3);

        $this->actingAs($user, 'api')
            ->postJson('/api/registered_animals')
            ->assertStatus(403)
            ->assertJson(['error' => 'account_not_premium']);
    }

    public function testStoreRegisteredAnimalWithOrganization()
    {
        $user = User::find(1);
        $user->account_type = User::ACCOUNT_TYPE_ORGANIZATION;
        $user->save();

        $this->actingAs($user, 'api')
            ->postJson('/api/registered_animals')
            ->assertStatus(403)
            ->assertJson(['error' => 'account_type_not_allowed']);
    }

    public function testStoreRegisteredAnimalWithOrganizationAccess()
    {
        $user = User::find(1);
        $user->account_type = User::ACCOUNT_TYPE_ACCESS;
        $user->save();

        $this->actingAs($user, 'api')
            ->postJson('/api/registered_animals')
            ->assertStatus(403)
            ->assertJson(['error' => 'account_type_not_allowed']);
    }

    public function testStoreRegisteredAnimalSuccessfully()
    {
        $user = User::find(1);

        $payload = [
            'name' => 'Medor',
            'animal_species' => Animal::ANIMAL_SPECIES_PET_DOG_SMALL,
            'identification_type' => Animal::IDENTIFICATION_TYPE_MICROCHIP,
            'identification_number' => '1234',
            'comment' => 'Comment',
            'images' => [
                UploadedFile::fake()->image('image1.jpg'),
                UploadedFile::fake()->image('image2.jpg'),
            ],
        ];

        $this->actingAs($user, 'api')
            ->postJson('/api/registered_animals', $payload)
            ->assertStatus(201)
            ->assertJson([
                'data' =>  [
                    'id' => 3,
                    'name' => 'Medor',
                    'animal_species' => Animal::ANIMAL_SPECIES_PET_DOG_SMALL,
                    'identification_type' => Animal::IDENTIFICATION_TYPE_MICROCHIP,
                    'identification_number' => '1234',
                    'comment' => 'Comment',
                    //'images' => [],
                    //'thumbnail' => '',
                ],
            ]);
    }

    /***************************************************************************
     * Test Update Registered Animal
     **************************************************************************/
    public function testUpdateRegisteredAnimalWithUnauthenticated()
    {
        $this->putJson('/api/registered_animals/2')
            ->assertStatus(401)
            ->assertJson(['message' => 'Unauthenticated.']);
    }

    public function testUpdateRegisteredAnimalWithForbidden()
    {
        $user = User::find(1);

        $payload = [
            'name' => 'Médor 2',
            'animal_species' => Animal::ANIMAL_SPECIES_PET_DOG_SMALL,
            'identification_type' => Animal::IDENTIFICATION_TYPE_RING,
            'identification_number' => '4321',
            'images' => ['image_1'],
        ];

        $this->actingAs($user, 'api')
            ->putJson('/api/registered_animals/2', $payload)
            ->assertStatus(403)
            ->assertJson(['message' => 'This action is unauthorized.']);
    }

    public function testUpdateRegisteredAnimalWithNoPremium()
    {
        $user = User::find(3);

        $this->actingAs($user, 'api')
            ->putJson('/api/registered_animals/2')
            ->assertStatus(403)
            ->assertJson(['error' => 'account_not_premium']);
    }

    public function testUpdateRegisteredAnimalWithOrganization()
    {
        $user = User::find(1);
        $user->account_type = User::ACCOUNT_TYPE_ORGANIZATION;
        $user->save();

        $this->actingAs($user, 'api')
            ->putJson('/api/registered_animals/2')
            ->assertStatus(403)
            ->assertJson(['error' => 'account_type_not_allowed']);
    }

    public function testUpdateRegisteredAnimalWithOrganizationAccess()
    {
        $user = User::find(1);
        $user->account_type = User::ACCOUNT_TYPE_ACCESS;
        $user->save();

        $this->actingAs($user, 'api')
            ->putJson('/api/registered_animals/2')
            ->assertStatus(403)
            ->assertJson(['error' => 'account_type_not_allowed']);
    }

    public function testUpdateRegisteredAnimalSuccessfully()
    {
        $user = User::find(1);

        $payload = [
            'name' => 'Medor 2',
            'animal_species' => Animal::ANIMAL_SPECIES_PET_DOG_SMALL,
            'identification_type' => Animal::IDENTIFICATION_TYPE_RING,
            'identification_number' => '4321',
            'comment' => 'Comment 2',
            'images' => ['image_1'],
        ];

        $this->actingAs($user, 'api')
            ->putJson('/api/registered_animals/1', $payload)
            ->assertStatus(200)
            ->assertJson([
                'data' => [
                    'id' => 1,
                    'name' => 'Medor 2',
                    'animal_species' => Animal::ANIMAL_SPECIES_PET_DOG_SMALL,
                    'identification_type' => Animal::IDENTIFICATION_TYPE_RING,
                    'identification_number' => '4321',
                    'comment' => 'Comment 2',
                    //'images' => [],
                    //'thumbnail' => '',
                ],
            ]);
    }

    /***************************************************************************
     * Test Delete Registered Animal
     **************************************************************************/
    public function testDeleteRegisteredAnimalWithUnauthenticated()
    {
        $this->deleteJson('/api/registered_animals/2')
            ->assertStatus(401)
            ->assertJson(['message' => 'Unauthenticated.']);
    }

    public function testDeleteRegisteredAnimalWithForbidden()
    {
        $user = User::find(1);

        $this->actingAs($user, 'api')
            ->deleteJson('/api/registered_animals/2')
            ->assertStatus(403)
            ->assertJson(['message' => 'This action is unauthorized.']);
    }

    public function testDeleteRegisteredAnimalWithNoPremium()
    {
        $user = User::find(3);

        $this->actingAs($user, 'api')
            ->deleteJson('/api/registered_animals/2')
            ->assertStatus(403)
            ->assertJson(['error' => 'account_not_premium']);
    }

    public function testDeleteRegisteredAnimalWithOrganization()
    {
        $user = User::find(1);
        $user->account_type = User::ACCOUNT_TYPE_ORGANIZATION;
        $user->save();

        $this->actingAs($user, 'api')
            ->deleteJson('/api/registered_animals/2')
            ->assertStatus(403)
            ->assertJson(['error' => 'account_type_not_allowed']);
    }

    public function testDeleteRegisteredAnimalWithOrganizationAccess()
    {
        $user = User::find(1);
        $user->account_type = User::ACCOUNT_TYPE_ACCESS;
        $user->save();

        $this->actingAs($user, 'api')
            ->deleteJson('/api/registered_animals/2')
            ->assertStatus(403)
            ->assertJson(['error' => 'account_type_not_allowed']);
    }

    public function testDeleteRegisteredAnimalSuccessfully()
    {
        $user = User::find(1);

        $this->actingAs($user, 'api')
            ->deleteJson('/api/registered_animals/1')
            ->assertStatus(204);
    }

    // End of class
}
