<?php

namespace Modules\Api\Http\Services;

use Modules\Common\Entities\Report;

class ReportService
{

    /***************************************************************************
     * Constants
     **************************************************************************/
    private const PAGINATION_PER_PAGE = 25;
    private const PAGINATION_FOR_MAP_PER_PAGE = 250;
    private const DEFAULT_RADIUS = 100;

    /***************************************************************************
     * Methods
     **************************************************************************/
    public function index(array $attributes)
    {
        // Prepare query
        $query = Report::available()->with('user', 'media')->latest('expires_at');

        // Apply location filter
        $query = $this->applyLocationFilter($attributes, $query);

        // Apply type filter
        $query = $this->applyTypeFilter($attributes, $query);

        // Apply animal condition filter
        $query = $this->applyAnimalConditionFilter($attributes, $query);

        // Apply animal species filter
        $query = $this->applyAnimalSpeciesFilter($attributes, $query);

        // Get pagination per page
        $paginationPerPage = $this->getPaginationPerPage($attributes);

        // Get reports
        $reports = $query->paginate($paginationPerPage);

        // Return reports
        return $reports;
    }

    public function store(array $attributes)
    {
        // Create report
        $report = auth()->user()->reports()->create($attributes);

        // Set expiration
        $report->setExpiration();

        // Return report
        return $report;
    }

    public function delete(array $attributes, Report $report)
    {
        // Handle deletion reason
        $attributes['deletion_reason'] = $attributes['deletion_reason'] ?? Report::DELETION_REASON_OTHER;
        $report->update($attributes);

        // Delete report
        $report->delete();
    }

    /***************************************************************************
     * Utils
     **************************************************************************/
    private function applyLocationFilter(array $attributes, $query)
    {
        // Get location
        $latitude = $attributes['latitude'] ?? null;
        $longitude = $attributes['longitude'] ?? null;
        $radius = $attributes['radius'] ?? null;

        // If exists then apply filter
        if ($latitude && $longitude && $radius) {
            $query = $query->inRadius([
                'latitude' => $latitude,
                'longitude' => $longitude,
            ], $radius);
        }

        // Return query
        return $query;
    }

    private function applyTypeFilter(array $attributes, $query)
    {
        // Get type
        $type = $attributes['type'] ?? null;

        // If exists then apply filter
        if ($type) {
            $query = $query->where('reports.type', '=', $type);
        }

        // Return query
        return $query;
    }

    private function applyAnimalConditionFilter(array $attributes, $query)
    {
        // Get animal condition
        $animalCondition = $attributes['animal_condition'] ?? null;

        // If exists then apply filter
        if ($animalCondition) {
            $query = $query->where('reports.animal_condition', '=', $animalCondition);
        }

        // Return query
        return $query;
    }

    private function applyAnimalSpeciesFilter(array $attributes, $query)
    {
        // Get animal species
        $animalSpecies = $attributes['animal_species'] ?? null;

        // If exists then apply filter
        if ($animalSpecies) {
            $query = $query->where('reports.animal_species', 'like', $animalSpecies . '%');
        }

        // Return query
        return $query;
    }

    private function getPaginationPerPage(array $attributes)
    {
        // Get map attribute
        $map = $attributes['map'] ?? false;

        // Return result
        return $map == 'true' ? self::PAGINATION_FOR_MAP_PER_PAGE : self::PAGINATION_PER_PAGE;
    }

    // End of class
}
