<?php

namespace Modules\Api\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Str;
use App\Http\Controllers\Controller;
use Modules\Api\Http\Resources\RegisteredAnimal as RegisteredAnimalResource;
use Modules\Api\Http\Services\RegisteredAnimalService;
use Modules\Common\Entities\RegisteredAnimal;
use Modules\Common\Http\Requests\RegisteredAnimalStoreRequest;
use Modules\Common\Http\Requests\RegisteredAnimalUpdateRequest;

class RegisteredAnimalController extends Controller
{

    /***************************************************************************
     * Attributes
     **************************************************************************/
    protected $registeredAnimalService;

    /***************************************************************************
     * Constructors
     **************************************************************************/
    public function __construct(RegisteredAnimalService $registeredAnimalService)
    {
        $this->registeredAnimalService = $registeredAnimalService;
    }

    /***************************************************************************
     * Methods
     **************************************************************************/
    public function show(RegisteredAnimal $registeredAnimal)
    {
        // Authorize user
        $this->authorize('write', $registeredAnimal);

        // Return resource
        return new RegisteredAnimalResource($registeredAnimal);
    }

    public function store(RegisteredAnimalStoreRequest $request)
    {
        // Create resource
        $registeredAnimal = $this->registeredAnimalService->store($request->validated());

        // Handle images
        $this->handleImages($request, $registeredAnimal);

        // Refresh resource
        $registeredAnimal->refresh();

        // Return response
        return new RegisteredAnimalResource($registeredAnimal);
    }

    public function update(RegisteredAnimalUpdateRequest $request, RegisteredAnimal $registeredAnimal)
    {
        // Authorize user
        $this->authorize('write', $registeredAnimal);

        // Update resource
        $registeredAnimal->update($request->validated());

        // Update images
        $this->updateImages($request, $registeredAnimal);

        // Handle images
        $this->handleImages($request, $registeredAnimal);

        // Refresh resource
        $registeredAnimal->refresh();

        // Return response
        return new RegisteredAnimalResource($registeredAnimal);
    }

    public function delete(RegisteredAnimal $registeredAnimal)
    {
        // Authorize user
        $this->authorize('write', $registeredAnimal);

        // Delete resource
        $registeredAnimal->delete();

        // Return response
        return response()->noContent();
    }

    /***************************************************************************
     * Utils
     **************************************************************************/
    private function handleImages(Request $request, RegisteredAnimal $registeredAnimal)
    {
        if ($request->hasFile('images')) {
            foreach ($request->file('images') as $image) {
                $registeredAnimal->addMedia($image)
                    ->usingFileName(Str::random(60) . '.jpg')
                    ->toMediaCollection();
            }
        }
    }

    private function updateImages(Request $request, RegisteredAnimal $registeredAnimal)
    {
        // Prepare images to remain
        $remainImages = [];
        if ($request->has('images')) {
            try {
                foreach ($request->get('images') as $image) {
                    if (is_string($image)) {
                        $remainImages[] = $image;
                    }
                }
            } catch (\Throwable $e) {
                // Handle exception
            }
        }

        // Delete images not needed anymore
        foreach ($registeredAnimal->getMedia() as $media) {
            // By url
            $mediaUrl = $media->getFullUrl('medium');
            if (!in_array($mediaUrl, $remainImages)) {
                $media->delete();
            }
        }
    }

    // End of class
}
