<?php

namespace Modules\Admin\Http\Controllers;

use Backpack\CRUD\app\Http\Controllers\CrudController;
use Backpack\CRUD\app\Library\CrudPanel\CrudPanelFacade as CRUD;
use Modules\Admin\Http\Controllers\Traits\ReportsExportOperation;
use Modules\Admin\Http\Controllers\Traits\RestoreOperation;
use Modules\Admin\Http\Requests\UpdateReportRequest;

use Modules\Common\Entities\Animal;
use Modules\Common\Entities\Report;

class ReportCrudController extends CrudController
{
    use \Backpack\CRUD\app\Http\Controllers\Operations\ListOperation;
    use \Backpack\CRUD\app\Http\Controllers\Operations\ShowOperation {
        show as traitShow;
    }
    use \Backpack\CRUD\app\Http\Controllers\Operations\UpdateOperation;
    use \Backpack\CRUD\app\Http\Controllers\Operations\DeleteOperation;
    use ReportsExportOperation;
    use RestoreOperation;

    /***************************************************************************
     * Setup
     **************************************************************************/
    public function setup()
    {
        CRUD::setModel("Modules\Common\Entities\Report");
        CRUD::setRoute(config('backpack.base.route_prefix') . '/reports');
        CRUD::setEntityNameStrings(__('messages.global.report'), __('messages.global.reports'));
        CRUD::set('show.setFromDb', false);
    }

    /***************************************************************************
     * List
     **************************************************************************/
    public function setupListOperation()
    {
        CRUD::addButtonFromView('top', 'export', 'export');
        CRUD::addColumns($this->fields());
        CRUD::removeColumns([
            'user_id', 'images', 'location_address', 'location_latitude', 'location_longitude',
            'final_location', 'comment', 'identification_type',
            'identification_number', 'updated_at', 'state',
            'deletion_reason_comment', 'deleted_at',
        ]);
        CRUD::addColumn([
            'name' => 'has_expired',
            'type' => 'boolean',
            'label' => __('messages.global.has_expired'),
            'wrapper' => [
                'element' => 'span',
                'class' => function ($crud, $column, $entry, $related_key) {
                    return $related_key ? 'badge badge-danger' : 'badge badge-success';
                },
            ],
            'orderable'  => true,
            'orderLogic' => function ($query, $column, $columnDirection) {
                return $query->orderBy('expires_at', $columnDirection);
            }
        ]);

        CRUD::addFilter(
            [
                'type'  => 'text',
                'name'  => 'email',
                'label' => __('messages.global.user'),
            ],
            false,
            function ($value) {
                return $this->crud->query->whereHas('user', function ($q) use ($value) {
                    $q->where('email', 'like', '%' . $value . '%');
                });
            }
        );

        $this->addRestoreFilter();
    }

    /***************************************************************************
     * Show
     **************************************************************************/
    public function setupShowOperation()
    {
        CRUD::addColumn([
            'name' => 'has_expired',
            'type' => 'boolean',
            'label' => __('messages.global.has_expired'),
            'wrapper' => [
                'element' => 'span',
                'class' => function ($crud, $column, $entry, $related_key) {
                    return $related_key ? 'badge badge-danger' : 'badge badge-success';
                },
            ],
        ]);
        CRUD::addColumn([
            'name' => 'state',
            'type' => 'select_from_array',
            'label' => __('messages.global.state'),
            'options' => [
                Report::STATE_WAITING => __('messages.report_state.' . Report::STATE_WAITING),
                Report::STATE_SUSPENDED => __('messages.report_state.' . Report::STATE_SUSPENDED),
                Report::STATE_DONE => __('messages.report_state.' . Report::STATE_DONE),
            ],
            'wrapper' => [
                'element' => 'span',
                'class' => function ($crud, $column, $entry, $related_key) {
                    if ($related_key == Report::STATE_WAITING) return 'badge badge-success';
                    if ($related_key == Report::STATE_SUSPENDED) return 'badge badge-warning';
                    return 'badge';
                },
            ],
            'orderable'  => true,
            'orderLogic' => function ($query, $column, $columnDirection) {
                return $query->orderBy('state', $columnDirection);
            }
        ]);
        CRUD::addColumns($this->fields());
    }

    public function show($id)
    {
        $this->crud->entry = $this->crud->model->withTrashed()->findOrFail($id);
        $this->crud->entry = $this->crud->entry->withFakes();

        return $this->traitShow($id);
    }

    /***************************************************************************
     * Update
     **************************************************************************/
    public function setupUpdateOperation()
    {
        CRUD::setValidation(UpdateReportRequest::class);
        CRUD::addFields($this->fields());
        CRUD::removeFields([
            'images', 'created_at', 'updated_at',
        ]);
    }

    /***************************************************************************
     * Fields
     **************************************************************************/
    private function fields()
    {
        return [
            [
                'name' => 'id',
                'type' => 'number',
                'label' => '#',
            ],
            [
                'name' => 'user_id',
                'type' => 'select',
                'label' => __('messages.global.user'),
                'attribute' => 'email',
                'entity' => 'user',
                'model' => 'Modules\Common\Entities\User',
            ],
            [
                'name' => 'type',
                'type' => 'select_from_array',
                'label' => __('messages.global.report_type'),
                'allows_null' => true,
                'options' => [
                    Report::TYPE_LOST => __('messages.report_type.' . Report::TYPE_LOST),
                    Report::TYPE_FOUND => __('messages.report_type.' . Report::TYPE_FOUND),
                ],
            ],
            [
                'name' => 'animal_species',
                'type' => 'select_from_array',
                'label' => __('messages.global.animal_species'),
                'allows_null' => true,
                'options' => $this->buildAnimalSpeciesOptions(),
            ],
            [
                'name' => 'animal_condition',
                'type' => 'select_from_array',
                'label' => __('messages.global.animal_condition'),
                'allows_null' => true,
                'options' => [
                    Animal::ANIMAL_CONDITION_HEALTHY => __('messages.animal_condition.' . Animal::ANIMAL_CONDITION_HEALTHY),
                    Animal::ANIMAL_CONDITION_HURT => __('messages.animal_condition.' . Animal::ANIMAL_CONDITION_HURT),
                    Animal::ANIMAL_CONDITION_DEAD => __('messages.animal_condition.' . Animal::ANIMAL_CONDITION_DEAD),
                ],
            ],
            [
                'name' => 'identification_type',
                'type' => 'select_from_array',
                'label' => __('messages.global.identification_type'),
                'allows_null' => true,
                'options' => [
                    Animal::IDENTIFICATION_TYPE_MICROCHIP => __('messages.identification_type.' . Animal::IDENTIFICATION_TYPE_MICROCHIP),
                    Animal::IDENTIFICATION_TYPE_RING => __('messages.identification_type.' . Animal::IDENTIFICATION_TYPE_RING),
                    Animal::IDENTIFICATION_TYPE_EARRING => __('messages.identification_type.' . Animal::IDENTIFICATION_TYPE_EARRING),
                    Animal::IDENTIFICATION_TYPE_TATTOO => __('messages.identification_type.' . Animal::IDENTIFICATION_TYPE_TATTOO),
                ],
            ],
            [
                'name' => 'identification_number',
                'type' => 'text',
                'label' => __('messages.global.identification_number'),
            ],
            [
                'name' => 'final_location',
                'type' => 'text',
                'label' => __('messages.global.where_is_animal'),
            ],
            [
                'name' => 'comment',
                'type' => 'textarea',
                'label' => __('messages.global.comment'),
            ],
            [
                'name' => 'location_address',
                'type' => 'text',
                'label' => __('messages.global.address'),
            ],
            [
                'name' => 'location_zipcode',
                'type' => 'text',
                'label' => __('messages.global.zipcode'),
            ],
            [
                'name' => 'location_country',
                'type' => 'text',
                'label' => __('messages.global.country'),
            ],
            [
                'name' => 'location_latitude',
                'type' => 'number',
                'label' => __('messages.global.latitude'),
                'decimals' => 11,
                'attributes' => ['step' => 'any'],
            ],
            [
                'name' => 'location_longitude',
                'type' => 'number',
                'label' => __('messages.global.longitude'),
                'decimals' => 11,
                'attributes' => ['step' => 'any'],
            ],
            [
                'name' => 'created_at',
                'type' => 'datetime',
                'label' => __('messages.global.created_at'),
            ],
            [
                'name' => 'updated_at',
                'type' => 'datetime',
                'label' => __('messages.global.updated_at'),
            ],
            [
                'name' => 'expires_at',
                'type' => 'datetime',
                'label' => __('messages.global.expires_at'),
            ],
            [
                'name' => 'deleted_at',
                'type' => 'datetime',
                'label' => __('messages.global.deleted_at'),
            ],
            [
                'name' => 'images',
                'type' => 'upload_multiple',
                'label' => __('messages.global.images'),
            ],
            [
                'name' => 'state',
                'type' => 'select_from_array',
                'label' => __('messages.global.state'),
                'options' => [
                    Report::STATE_WAITING => __('messages.report_state.' . Report::STATE_WAITING),
                    Report::STATE_SUSPENDED => __('messages.report_state.' . Report::STATE_SUSPENDED),
                    Report::STATE_DONE => __('messages.report_state.' . Report::STATE_DONE),
                ],
            ],
            [
                'name' => 'deletion_reason',
                'type' => 'select_from_array',
                'label' => __('messages.global.deletion_reason'),
                'allows_null' => true,
                'options' => [
                    Report::DELETION_REASON_FOUND => __('messages.deletion_reason.' . Report::DELETION_REASON_FOUND),
                    Report::DELETION_REASON_SECURE => __('messages.deletion_reason.' . Report::DELETION_REASON_SECURE),
                    Report::DELETION_REASON_DEAD => __('messages.deletion_reason.' . Report::DELETION_REASON_DEAD),
                    Report::DELETION_REASON_MISTAKE => __('messages.deletion_reason.' . Report::DELETION_REASON_MISTAKE),
                    Report::DELETION_REASON_OTHER => __('messages.deletion_reason.' . Report::DELETION_REASON_OTHER),
                ],
            ],
            [
                'name' => 'deletion_reason_comment',
                'type' => 'text',
                'label' => __('messages.global.deletion_reason_comment'),
            ],
        ];
    }

    private function buildAnimalSpeciesOptions()
    {
        $species = [];

        foreach (Animal::ANIMAL_SPECIES as $value) {
            $species[$value] = __('messages.animal_species.' . $value);
        }

        return $species;
    }

    // End of class
}
